const fs = require('fs');
const os = require('os');
const path = require('path');
const logger = require('./logger');
const { extractAudioToPcm, extractFullAudioToFile, AUDIO_SAMPLE_RATE } = require('./ffmpeg');

const SIMILARITY_THRESHOLD = 0.75;
const MIN_GAP_SECONDS = 2;
const MIN_TEMPLATE_DURATION = 0.3;
const STEP_SAMPLES = Math.floor(AUDIO_SAMPLE_RATE * 0.05);

function bufferToInt16Array(buffer) {
  const len = buffer.length / 2;
  const arr = new Int16Array(len);
  for (let i = 0; i < len; i++) {
    arr[i] = buffer.readInt16LE(i * 2);
  }
  return arr;
}

function computeMean(arr, start, length) {
  let sum = 0;
  for (let i = 0; i < length; i++) {
    sum += arr[start + i];
  }
  return sum / length;
}

function computeStd(arr, start, length, mean) {
  let sumSq = 0;
  for (let i = 0; i < length; i++) {
    const d = arr[start + i] - mean;
    sumSq += d * d;
  }
  return Math.sqrt(sumSq / length) || 1e-10;
}

function normalizedCrossCorrelation(ref, full, fullStart, n) {
  const meanRef = computeMean(ref, 0, n);
  const meanWin = computeMean(full, fullStart, n);
  const stdRef = computeStd(ref, 0, n, meanRef);
  const stdWin = computeStd(full, fullStart, n, meanWin);

  let sum = 0;
  for (let i = 0; i < n; i++) {
    sum += (ref[i] - meanRef) * (full[fullStart + i] - meanWin);
  }
  return sum / (n * stdRef * stdWin);
}

/**
 * Find audio template matches using normalized cross-correlation.
 * @param {Int16Array} ref - Reference template samples
 * @param {Int16Array} full - Full audio samples
 * @param {number} sampleRate - Sample rate
 * @param {Object} options - { similarityThreshold, minGapSeconds, stepSamples }
 * @returns {Array<{startTime: number}>}
 */
function findMatches(ref, full, sampleRate, options = {}) {
  const threshold = options.similarityThreshold ?? SIMILARITY_THRESHOLD;
  const minGapSamples = Math.floor((options.minGapSeconds ?? MIN_GAP_SECONDS) * sampleRate);
  const step = 100; // Check every 100 samples (~6ms) for balance of speed and accuracy
  const templateStart = options.templateStart || 0;

  const n = ref.length;
  const matches = [];
  let lastMatchEnd = -minGapSamples - 1;
  let maxNcc = 0;
  let maxNccPos = -1;

  const loopCount = Math.floor((full.length - n) / step) + 1;
  logger.log(`[AudioHighlight] Starting main loop: ${loopCount} iterations, threshold=${threshold}`);

  for (let i = 0; i <= full.length - n; i += step) {
    const ncc = normalizedCrossCorrelation(ref, full, i, n);
    if (ncc > maxNcc) {
      maxNcc = ncc;
      maxNccPos = i;
    }
    if (ncc >= threshold && i - lastMatchEnd >= minGapSamples) {
      logger.log(`[AudioHighlight] MATCH at position ${i} (${(i / sampleRate).toFixed(2)}s), NCC=${ncc.toFixed(4)}`);
      const startTime = i / sampleRate;
      matches.push({ startTime });
      lastMatchEnd = i + n;
      i += n - step;
    }
  }

  logger.log(`[AudioHighlight] Max NCC overall: ${maxNcc.toFixed(4)} at position ${maxNccPos} (${(maxNccPos / sampleRate).toFixed(2)}s)`);

  return matches;
}

/**
 * Detect highlights by matching a template audio clip against the full video.
 * @param {Object} params
 * @param {string} params.videoPath - Path to the video file
 * @param {number} params.templateStart - Template start time in seconds
 * @param {number} params.templateEnd - Template end time in seconds
 * @param {number} [params.similarityThreshold] - NCC threshold (0-1)
 * @param {number} [params.minGapSeconds] - Minimum gap between matches
 * @param {Function} [params.onProgress] - Callback (percent, message) for progress updates
 * @returns {Promise<{success: boolean, matches?: Array<{startTime, endTime?, title}>, error?: string}>}
 */
async function detectHighlights({ videoPath, templateStart, templateEnd, similarityThreshold, minGapSeconds, onProgress }) {
  const report = (p, msg) => {
    if (typeof onProgress === 'function') onProgress(p, msg);
    logger.log(`[AudioHighlight] ${msg} (${Math.round(p)}%)`);
  };

  try {
    if (!videoPath || !fs.existsSync(videoPath)) {
      return { success: false, error: 'Video file not found.' };
    }

    const duration = templateEnd - templateStart;
    if (duration < MIN_TEMPLATE_DURATION) {
      return { success: false, error: `Template too short. Use at least ${MIN_TEMPLATE_DURATION} seconds.` };
    }

    report(5, 'Extracting template audio...');
    const refBuffer = await extractAudioToPcm(videoPath, templateStart, templateEnd);
    if (!refBuffer || refBuffer.length < 1024) {
      return { success: false, error: 'No audio track or extraction failed.' };
    }

    report(25, 'Extracting full video audio...');
    const tempDir = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'miba-audio-'));
    const fullPcmPath = path.join(tempDir, 'full.pcm');

    try {
      await extractFullAudioToFile(videoPath, fullPcmPath);
    } catch (extractError) {
      await fs.promises.rm(tempDir, { recursive: true, force: true }).catch(() => {});
      return { success: false, error: extractError.message || 'Failed to extract audio from video.' };
    }

    report(50, 'Loading audio for analysis...');
    const fullBuffer = await fs.promises.readFile(fullPcmPath);
    await fs.promises.rm(tempDir, { recursive: true, force: true }).catch(() => {});

    const ref = bufferToInt16Array(refBuffer);
    const full = bufferToInt16Array(fullBuffer);

    logger.log(`[AudioHighlight] Template samples: ${ref.length}, Full audio samples: ${full.length}`);
    logger.log(`[AudioHighlight] Template: ${templateStart}s - ${templateEnd}s (${(templateEnd - templateStart).toFixed(2)}s), Full video duration ~${(full.length / AUDIO_SAMPLE_RATE).toFixed(1)}s`);

    // Debug: check NCC at the exact template position
    const templateStartSample = Math.floor(templateStart * AUDIO_SAMPLE_RATE);
    if (templateStartSample > 0 && templateStartSample + ref.length <= full.length) {
      const debugNcc = normalizedCrossCorrelation(ref, full, templateStartSample, ref.length);
      logger.log(`[AudioHighlight] Debug NCC at template position (${templateStart}s): ${debugNcc.toFixed(4)}`);
    }

    report(70, 'Scanning for matches...');
    const matches = findMatches(ref, full, AUDIO_SAMPLE_RATE, {
      similarityThreshold,
      minGapSeconds,
      templateStart
    });

    report(100, `Found ${matches.length} match(es).`);

    const cuePointMatches = matches.map((m) => ({
      startTime: m.startTime,
      endTime: m.startTime + duration,
      title: 'Highlight',
      source: 'audio-scan'
    }));

    return { success: true, matches: cuePointMatches };
  } catch (error) {
    logger.error('[AudioHighlight] Error:', error);
    return {
      success: false,
      error: error.message || 'Audio highlight detection failed.'
    };
  }
}

/**
 * Detect highlights using a pre-extracted audio sample (e.g. from a saved template).
 * @param {Object} params
 * @param {string} params.videoPath - Path to the video file to scan
 * @param {string} params.audioSampleBase64 - Base64-encoded PCM buffer of the template sample
 * @param {number} [params.similarityThreshold] - NCC threshold (0-1)
 * @param {number} [params.minGapSeconds] - Minimum gap between matches
 * @param {Function} [params.onProgress] - Callback (percent, message) for progress updates
 * @returns {Promise<{success: boolean, matches?: Array<{startTime, endTime?, title}>, error?: string}>}
 */
async function detectHighlightsWithTemplate({ videoPath, audioSampleBase64, similarityThreshold, minGapSeconds, onProgress }) {
  const report = (p, msg) => {
    if (typeof onProgress === 'function') onProgress(p, msg);
    logger.log(`[AudioHighlight] ${msg} (${Math.round(p)}%)`);
  };

  try {
    if (!videoPath || !fs.existsSync(videoPath)) {
      return { success: false, error: 'Video file not found.' };
    }
    if (!audioSampleBase64 || typeof audioSampleBase64 !== 'string') {
      return { success: false, error: 'Audio sample is required.' };
    }

    const refBuffer = Buffer.from(audioSampleBase64, 'base64');
    if (refBuffer.length < 1024) {
      return { success: false, error: 'Template sample too short.' };
    }

    const duration = refBuffer.length / (2 * AUDIO_SAMPLE_RATE);

    report(10, 'Extracting full video audio...');
    const tempDir = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'miba-audio-'));
    const fullPcmPath = path.join(tempDir, 'full.pcm');

    try {
      await extractFullAudioToFile(videoPath, fullPcmPath);
    } catch (extractError) {
      await fs.promises.rm(tempDir, { recursive: true, force: true }).catch(() => {});
      return { success: false, error: extractError.message || 'Failed to extract audio from video.' };
    }

    report(50, 'Loading audio for analysis...');
    const fullBuffer = await fs.promises.readFile(fullPcmPath);
    await fs.promises.rm(tempDir, { recursive: true, force: true }).catch(() => {});

    const ref = bufferToInt16Array(refBuffer);
    const full = bufferToInt16Array(fullBuffer);

    logger.log(`[AudioHighlight] Template samples: ${ref.length}, Full audio samples: ${full.length}`);
    logger.log(`[AudioHighlight] Template duration ~${duration.toFixed(2)}s, Full video duration ~${(full.length / AUDIO_SAMPLE_RATE).toFixed(1)}s`);

    report(70, 'Scanning for matches...');
    const matches = findMatches(ref, full, AUDIO_SAMPLE_RATE, {
      similarityThreshold,
      minGapSeconds
    });

    report(100, `Found ${matches.length} match(es).`);

    const cuePointMatches = matches.map((m) => ({
      startTime: m.startTime,
      endTime: m.startTime + duration,
      title: 'Highlight',
      source: 'audio-scan'
    }));

    return { success: true, matches: cuePointMatches };
  } catch (error) {
    logger.error('[AudioHighlight] Error:', error);
    return {
      success: false,
      error: error.message || 'Audio highlight detection failed.'
    };
  }
}

const WAVEFORM_BINS = 4000;

/**
 * Get downsampled waveform data for visualization.
 * @param {string} videoPath - Path to the video file
 * @param {number} duration - Video duration in seconds (for scaling)
 * @returns {Promise<{success: boolean, peaks?: number[], duration?: number, error?: string}>}
 */
async function getWaveformData(videoPath, duration) {
  const tempDir = path.join(os.tmpdir(), 'miba-waveform-');
  let tempDirPath = null;

  try {
    if (!videoPath || !fs.existsSync(videoPath)) {
      return { success: false, error: 'Video file not found.' };
    }

    tempDirPath = await fs.promises.mkdtemp(tempDir);
    const fullPcmPath = path.join(tempDirPath, 'full.pcm');

    await extractFullAudioToFile(videoPath, fullPcmPath);
    const buffer = await fs.promises.readFile(fullPcmPath);
    await fs.promises.rm(tempDirPath, { recursive: true, force: true }).catch(() => {});

    const samples = bufferToInt16Array(buffer);
    const bins = Math.min(WAVEFORM_BINS, Math.floor(samples.length / 10));
    const binSize = samples.length / bins;
    const peaks = [];

    for (let i = 0; i < bins; i++) {
      const start = Math.floor(i * binSize);
      const end = Math.min(Math.floor((i + 1) * binSize), samples.length);
      let max = 0;
      for (let j = start; j < end; j++) {
        const v = Math.abs(samples[j]);
        if (v > max) max = v;
      }
      peaks.push(max / 32768);
    }

    return { success: true, peaks, duration: duration || 0 };
  } catch (error) {
    if (tempDirPath) {
      await fs.promises.rm(tempDirPath, { recursive: true, force: true }).catch(() => {});
    }
    logger.error('[Waveform] Error:', error);
    return { success: false, error: error.message || 'Failed to extract waveform.' };
  }
}

async function detectCombinedHighlights({ videoPath, audioTemplateBase64, visualTemplateBase64, visualRegion, audioThreshold, visualThreshold, method, frameSkip, batchSize, minGapSeconds, onProgress }) {
  const report = (p, msg) => {
    if (typeof onProgress === 'function') onProgress(p, msg);
    logger.log(`[CombinedHighlight] ${msg} (${Math.round(p)}%)`);
  };

  try {
    if (!videoPath) {
      return { success: false, error: 'Video file path is required.' };
    }

    const hasAudio = audioTemplateBase64 && audioTemplateBase64.length > 100;
    const hasVisual = visualTemplateBase64 && visualRegion;

    if (!hasAudio && !hasVisual) {
      return { success: false, error: 'At least one of audio or visual template is required.' };
    }

    report(5, 'Starting combined detection...');

    const audioPromise = hasAudio ? detectHighlightsWithTemplate({
      videoPath,
      audioSampleBase64: audioTemplateBase64,
      similarityThreshold: audioThreshold,
      minGapSeconds,
      onProgress: (p, m) => report(p * 0.4, `[Audio] ${m}`)
    }) : Promise.resolve({ success: true, matches: [] });

    const visualPromise = hasVisual ? (async () => {
      const { detectVisualHighlights } = require('./visualHighlightDetector');
      return await detectVisualHighlights({
        videoPath,
        templateBase64: visualTemplateBase64,
        region: visualRegion,
        method: method || 'fast',
        threshold: visualThreshold,
        frameSkip: frameSkip || 5,
        batchSize: batchSize || 128,
        minGapSeconds: minGapSeconds || 2,
        onProgress: (p, m) => report(40 + p * 0.4, `[Visual] ${m}`)
      });
    })() : Promise.resolve({ success: true, matches: [] });

    const [audioResult, visualResult] = await Promise.all([audioPromise, visualPromise]);

    const audioMatches = audioResult.success ? audioResult.matches : [];
    const visualMatches = visualResult.success ? visualResult.matches : [];

    report(90, 'Combining results...');

    const combinedMatches = [];
    const usedVisual = new Set();

    for (const audioMatch of audioMatches) {
      const nearbyVisual = visualMatches.find(vm => 
        Math.abs(vm.startTime - audioMatch.startTime) < 3 &&
        !usedVisual.has(vm.startTime)
      );

      if (nearbyVisual) {
        combinedMatches.push({
          startTime: audioMatch.startTime,
          endTime: Math.max(audioMatch.endTime || audioMatch.startTime + 2, nearbyVisual.endTime || nearbyVisual.startTime + 2),
          title: 'Highlight (Audio + Visual)',
          source: 'combined-scan',
          audioMatch: true,
          visualMatch: true,
          audioSimilarity: audioMatch.similarity,
          visualSimilarity: nearbyVisual.similarity
        });
        usedVisual.add(nearbyVisual.startTime);
      } else if (hasAudio && hasVisual) {
        combinedMatches.push({
          startTime: audioMatch.startTime,
          endTime: audioMatch.endTime || audioMatch.startTime + 2,
          title: 'Highlight (Audio only)',
          source: 'combined-scan',
          audioMatch: true,
          visualMatch: false,
          audioSimilarity: audioMatch.similarity
        });
      }
    }

    for (const visualMatch of visualMatches) {
      if (!usedVisual.has(visualMatch.startTime) && hasAudio) {
        combinedMatches.push({
          startTime: visualMatch.startTime,
          endTime: visualMatch.endTime || visualMatch.startTime + 2,
          title: 'Highlight (Visual only)',
          source: 'combined-scan',
          audioMatch: false,
          visualMatch: true,
          visualSimilarity: visualMatch.similarity
        });
      }
    }

    combinedMatches.sort((a, b) => a.startTime - b.startTime);

    report(100, `Found ${combinedMatches.length} combined highlight(s).`);

    return { success: true, matches: combinedMatches };
  } catch (error) {
    logger.error('[CombinedHighlight] Error:', error);
    return { success: false, error: error.message || 'Combined highlight detection failed.' };
  }
}

module.exports = {
  detectHighlights,
  detectHighlightsWithTemplate,
  detectCombinedHighlights,
  getWaveformData,
  MIN_TEMPLATE_DURATION,
  SIMILARITY_THRESHOLD,
  MIN_GAP_SECONDS
};
